import textwrap
from typing import TYPE_CHECKING

import pytest

from debputy.lsp.languages.lsp_debian_changelog import _lint_debian_changelog
from debputy.lsprotocol.types import DiagnosticSeverity
from debputy.packages import DctrlParser
from debputy.plugin.api.feature_set import PluginProvidedFeatureSet
from lint_tests.lint_tutil import (
    LintWrapper,
    diag_range_to_text,
    standard_quick_fixes_for_diagnostic,
    apply_code_action_edits,
)

if TYPE_CHECKING:
    from lsprotocol import types
else:
    from debputy.lsprotocol import types


@pytest.fixture
def line_linter(
    debputy_plugin_feature_set: PluginProvidedFeatureSet,
    lint_dctrl_parser: DctrlParser,
) -> LintWrapper:
    return LintWrapper(
        "/nowhere/debian/changelog",
        _lint_debian_changelog,
        debputy_plugin_feature_set,
        lint_dctrl_parser,
    )


def _code_actions_for(
    uri: str,
    text_range: "types.Range",
) -> "types.CodeActionsParams":
    return types.CodeActionParams(
        types.TextDocumentIdentifier(uri),
        text_range,
        # Ignore the actual list of diagnostics, since we do not use them.
        types.CodeActionContext([]),
    )


def test_dch_lint(line_linter: LintWrapper) -> None:
    lines = textwrap.dedent(
        """\
    foo (0.2) unstable; urgency=medium

      * Renamed to foo
    
     -- Niels Thykier <niels@thykier.net>  Mon, 08 Apr 2024 16:00:00 +0000

    bar (0.2) unstable; urgency=medium

      * Initial release
    
     -- Niels Thykier <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000
    """
    ).splitlines(keepends=True)

    diagnostics = line_linter(lines)
    print(diagnostics)
    # Without a control file, this is fine
    assert not diagnostics

    line_linter.dctrl_lines = textwrap.dedent(
        """\
    Source: foo
    
    Package: something-else
    """
    ).splitlines(keepends=True)

    diagnostics = line_linter(lines)
    print(diagnostics)
    # Also fine, because d/control and d/changelog agrees
    assert not diagnostics

    line_linter.dctrl_lines = textwrap.dedent(
        """\
    Source: bar

    Package: something-else
    """
    ).splitlines(keepends=True)

    diagnostics = line_linter(lines)
    print(diagnostics)
    # This should be problematic though
    assert diagnostics and len(diagnostics) == 1
    diag = diagnostics[0]

    msg = (
        "The first entry must use the same source name as debian/control."
        ' Changelog uses: "foo" while d/control uses: "bar"'
    )
    assert diag.severity == DiagnosticSeverity.Error
    assert diag.message == msg
    assert f"{diag.range}" == "0:0-0:3"


def test_dch_lint_historical(line_linter: LintWrapper) -> None:
    nonsense = "very very very very very very very very very very very very very very "
    lines = textwrap.dedent(
        f"""\
    foo (0.4) unstable; urgency=medium

      * A {nonsense} long line about absolute nothing that should trigger a warning about length.

     -- Niels Thykier <niels@thykier.net>  Mon, 08 Apr 2024 16:00:00 +0000

    foo (0.3) unstable; urgency=medium

      * Another entry that is not too long.

     -- Niels Thykier <niels@thykier.net>  Thu, 04 Apr 2024 00:00:00 +0000

    foo (0.2) unstable; urgency=medium

      * A {nonsense}  long line about absolute nothing that should not trigger a warning about length.

     -- Niels Thykier <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000
    """
    ).splitlines(keepends=True)
    diagnostics = line_linter(lines)
    print(diagnostics)
    # This should be problematic though
    assert diagnostics and len(diagnostics) == 1
    diag = diagnostics[0]

    msg = "Line exceeds 82 characters"
    assert diag.severity == DiagnosticSeverity.Hint
    assert diag.message == msg
    assert f"{diag.range}" == "2:82-2:153"


def test_dch_lint_invalid_version(line_linter: LintWrapper) -> None:
    lines = textwrap.dedent(
        f"""\
    foo (a!0.3!a) unstable; urgency=medium

      * Another entry that is not too long.

     -- Niels Thykier <niels@thykier.net>  Thu, 04 Apr 2024 00:00:00 +0000

    foo (a+!) unstable; urgency=medium

      * Initial release

     -- Niels Thykier <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000
    """
    ).splitlines(keepends=True)
    diagnostics = line_linter(lines)
    print(diagnostics)
    assert diagnostics and len(diagnostics) == 3
    first_issue, second_issue, third_issue = diagnostics

    msg = "This part cannot be parsed as a valid Debian version"
    assert first_issue.severity == DiagnosticSeverity.Error
    assert first_issue.message == msg
    assert f"{first_issue.range}" == "0:5-0:7"

    msg = "This part cannot be parsed as a valid Debian version"
    assert second_issue.severity == DiagnosticSeverity.Error
    assert second_issue.message == msg
    assert f"{second_issue.range}" == "0:10-0:12"

    msg = 'Cannot parse "a+!" as a Debian version.'
    assert third_issue.severity == DiagnosticSeverity.Error
    assert third_issue.message == msg
    assert f"{third_issue.range}" == "6:5-6:8"


def test_dch_lint_version_typo_dfsg(line_linter: LintWrapper) -> None:
    lines = textwrap.dedent(
        f"""\
    foo (0.3+dsfg) unstable; urgency=medium

      * Another entry that is not too long.

     -- Niels Thykier <niels@thykier.net>  Thu, 04 Apr 2024 00:00:00 +0000

    foo (0.2) unstable; urgency=medium

      * Initial release

     -- Niels Thykier <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000
    """
    ).splitlines(keepends=True)
    diagnostics = line_linter(lines)
    print(diagnostics)
    assert diagnostics and len(diagnostics) == 1
    issue = diagnostics[0]

    msg = 'Typo of "dfsg" (Debian Free Software Guidelines)'
    assert issue.severity == DiagnosticSeverity.Hint
    assert issue.message == msg
    assert f"{issue.range}" == "0:9-0:13"
    assert diag_range_to_text(lines, issue.range) == "dsfg"


@pytest.mark.parametrize(
    "spacing,expected_range,highlighted_text",
    [
        (
            "",
            "4:36-4:38",
            ">T",
        ),
        (
            " ",
            "4:37-4:38",
            " ",
        ),
        (
            "\t\t",
            "4:37-4:39",
            "\t\t",
        ),
        (
            "   ",
            "4:37-4:40",
            "   ",
        ),
    ],
)
def test_dch_lint_signoff_line_email_date_spacing(
    line_linter: LintWrapper,
    spacing: str,
    expected_range: str,
    highlighted_text: str,
) -> None:
    lines = textwrap.dedent(
        f"""\
    foo (0.3) unstable; urgency=medium

      * Another entry that is not too long.

     -- Niels Thykier <niels@thykier.net>{spacing}Thu, 04 Apr 2024 00:00:00 +0000

    foo (0.2) unstable; urgency=medium

      * Initial release

     -- Niels Thykier <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000
    """
    ).splitlines(keepends=True)
    diagnostics = line_linter(lines)
    print(diagnostics)
    assert diagnostics and len(diagnostics) == 1
    issue = diagnostics[0]

    msg = "Must be exactly two spaces between email and sign off date"
    assert issue.severity == DiagnosticSeverity.Error
    assert issue.message == msg
    assert f"{issue.range}" == expected_range
    assert diag_range_to_text(lines, issue.range) == highlighted_text
    quick_fixes = standard_quick_fixes_for_diagnostic(
        line_linter.lint_state,
        _code_actions_for(line_linter.lint_state.doc_uri, issue.range),
        issue,
    )
    assert len(quick_fixes) == 1
    quick_fix = quick_fixes[0]
    assert isinstance(quick_fix, types.CodeAction)
    fixed_text = apply_code_action_edits(
        line_linter.lint_state.doc_uri,
        lines,
        quick_fix,
    )
    fixed_lines = fixed_text.splitlines(keepends=True)
    expected_line_post_fix = (
        " -- Niels Thykier <niels@thykier.net>  Thu, 04 Apr 2024 00:00:00 +0000\n"
    )
    assert fixed_lines[4] == expected_line_post_fix


@pytest.mark.parametrize(
    "spacing",
    [
        "",
        " ",
        "   ",
    ],
)
def test_dch_lint_signoff_missing_sign_off_date(
    line_linter: LintWrapper,
    spacing: str,
) -> None:
    lines = textwrap.dedent(
        f"""\
    foo (0.3) unstable; urgency=medium

      * Another entry that is not too long.

     -- Niels Thykier <niels@thykier.net>{spacing}

    foo (0.2) unstable; urgency=medium

      * Initial release

     -- Niels Thykier <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000
    """
    ).splitlines(keepends=True)
    diagnostics = line_linter(lines)
    print(diagnostics)
    assert diagnostics and len(diagnostics) == 1
    issue = diagnostics[0]

    msg = "Missing sign off date"
    assert issue.severity == DiagnosticSeverity.Error
    assert issue.message == msg
    # The line was always stripped when the test was written, so the trailing space
    # was never included in the range for that reason.
    assert f"{issue.range}" == f"4:0-4:37"
    expected_text = f" -- Niels Thykier <niels@thykier.net>"
    assert diag_range_to_text(lines, issue.range) == expected_text


def test_dch_lint_signoff_empty_sign_off(line_linter: LintWrapper) -> None:
    lines = textwrap.dedent(
        f"""\
    foo (0.3) unstable; urgency=medium

      * Another entry that is not too long.

     --

    foo (0.2) unstable; urgency=medium

      * Initial release

     -- Niels Thykier <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000
    """
    ).splitlines(keepends=True)
    diagnostics = line_linter(lines)
    print(diagnostics)
    assert diagnostics and len(diagnostics) == 1
    issue = diagnostics[0]

    msg = 'Missing "Name <email@example.com>"'
    assert issue.severity == DiagnosticSeverity.Error
    assert issue.message == msg
    assert f"{issue.range}" == f"4:0-4:3"
    expected_text = f" --"
    assert diag_range_to_text(lines, issue.range) == expected_text

    lines = textwrap.dedent(
        f"""\
    foo (0.3) unstable; urgency=medium

      * Another entry that is not too long.

     --Niels

    foo (0.2) unstable; urgency=medium

      * Initial release

     -- Niels Thykier <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000
    """
    ).splitlines(keepends=True)
    diagnostics = line_linter(lines)
    print(diagnostics)
    assert diagnostics and len(diagnostics) == 1
    issue = diagnostics[0]

    msg = 'Start of sign-off line should be " -- ".'
    assert issue.severity == DiagnosticSeverity.Error
    assert issue.message == msg
    assert f"{issue.range}" == f"4:0-4:8"
    expected_text = f" --Niels"
    assert diag_range_to_text(lines, issue.range) == expected_text


@pytest.mark.parametrize(
    "signoff_line,issue_range,severity,issue_msg,highlight_text,ignore_extra_diagnostics",
    [
        (
            " -- Niels Thykier niels@thykier.net  Mon, 01 Apr 2024 00:00:00 +0000",
            "4:4-4:35",
            types.DiagnosticSeverity.Error,
            'Missing opening "<" to start the email address after the name',
            "Niels Thykier niels@thykier.net",
            False,
        ),
        (
            " -- Niels Thykier niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000",
            "4:18-4:36",
            types.DiagnosticSeverity.Error,
            'Missing opening "<" to start the email address after the name',
            "niels@thykier.net>",
            False,
        ),
        (
            " -- Niels Thykier niels@thykier.net Mon, 01 Apr 2024 00:00:00 +0000",
            "4:4-4:67",
            types.DiagnosticSeverity.Error,
            'Missing opening "<" to start the email address after the name',
            "Niels Thykier niels@thykier.net Mon, 01 Apr 2024 00:00:00 +0000",
            False,
        ),
        (
            " -- Niels Thykier <niels@thykier.net Mon, 01 Apr 2024 00:00:00 +0000",
            "4:18-4:36",
            types.DiagnosticSeverity.Error,
            'Missing closing ">" to finish email address before the sign off date',
            "<niels@thykier.net",
            # This also triggers the missing "  " spaces between email and date.
            # The point is deliberately to test that a single space is sufficient
            # to delimit the email, so we cannot avoid the second error here.
            True,
        ),
        (
            " -- <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000",
            "4:3-4:4",
            types.DiagnosticSeverity.Error,
            "Missing name before email",
            " ",
            False,
        ),
        (
            " --  <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000",
            "4:4-4:5",
            types.DiagnosticSeverity.Error,
            "Missing name before email",
            " ",
            False,
        ),
        (
            " --   Niels<niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000",
            "4:4-4:11",
            types.DiagnosticSeverity.Warning,
            "Non-standard spacing around the name",
            "  Niels",
            False,
        ),
        (
            " -- Niels  <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000",
            "4:4-4:11",
            types.DiagnosticSeverity.Warning,
            "Non-standard spacing around the name",
            "Niels  ",
            False,
        ),
        (
            " --  Niels <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000",
            "4:4-4:11",
            types.DiagnosticSeverity.Warning,
            "Non-standard spacing around the name",
            " Niels ",
            False,
        ),
        (
            " -- Niels Thykier <niels@thykier.net>  Mon",
            "4:39-4:42",
            types.DiagnosticSeverity.Error,
            "Expected a date in RFC822 format (Tue, 12 Mar 2024 12:34:56 +0000)",
            "Mon",
            False,
        ),
        (
            " -- Niels Thykier <niels@thykier.net>  ???, 01 Apr 2024 00:00:00 +0000",
            "4:39-4:42",
            types.DiagnosticSeverity.Error,
            "Expected a three letter date here using US English format (Mon, Tue, ..., Sun)",
            "???",
            False,
        ),
        (
            " -- Niels Thykier <niels@thykier.net>  Mon 2025-05-01",
            "4:42-4:44",
            types.DiagnosticSeverity.Error,
            'Improper formatting of date. Expected ", " here, not " 2"',
            " 2",
            False,
        ),
        (
            " -- Niels Thykier <niels@thykier.net>  Mon, Golf",
            "4:44-4:48",
            types.DiagnosticSeverity.Error,
            'Unable to parse the date as a valid RFC822 date: Invalid date value or format "Golf"',
            "Golf",
            False,
        ),
        (
            " -- Niels Thykier <niels@thykier.net>  Tue, 01 Apr 2024 00:00:00 +0000",
            "4:39-4:42",
            types.DiagnosticSeverity.Warning,
            "The date was a Monday",
            "Tue",
            False,
        ),
    ],
)
def test_dch_lint_signoff_problems(
    line_linter: LintWrapper,
    signoff_line: str,
    issue_range: str,
    severity: "types.DiagnosticSeverity",
    issue_msg: str,
    highlight_text: str,
    ignore_extra_diagnostics: bool,
) -> None:
    lines = (
        textwrap.dedent(
            """\
    foo (0.3) unstable; urgency=medium

      * Another entry that is not too long.

    {signoff_line}

    foo (0.2) unstable; urgency=medium

      * Initial release

     -- Niels Thykier <niels@thykier.net>  Mon, 01 Apr 2024 00:00:00 +0000
    """
        )
        .format(signoff_line=signoff_line)
        .splitlines(keepends=True)
    )
    diagnostics = line_linter(lines)
    print(lines)
    print(diagnostics)
    assert diagnostics
    assert len(diagnostics) == 1 or ignore_extra_diagnostics
    issue = diagnostics[0]

    assert issue.severity == severity
    assert issue.message == issue_msg
    assert f"{issue.range}" == issue_range
    assert diag_range_to_text(lines, issue.range) == highlight_text
