# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""UI helpers for workspaces."""

from functools import cached_property

from django.urls import reverse

from debusine.db.context import context
from debusine.db.models import Workspace
from debusine.web.icons import Icons
from debusine.web.views.places import Place
from debusine.web.views.ui.base import UI


class WorkspaceUI(UI[Workspace]):
    """UI helpers for Workspace instances."""

    @cached_property
    def place(self) -> Place:
        """Return a place to show this workspace."""
        return Place(
            title=self.instance.name,
            long_title=f"Workspace {self.instance.name}",
            url=self.instance.get_absolute_url(),
            icon=Icons.WORKSPACE,
        )

    @cached_property
    def place_update(self) -> Place:
        """Return a place to update this workspace."""
        return Place(
            title=f"Configure workspace {self.instance.name}",
            url=self.instance.get_absolute_url_configure(),
        )

    @cached_property
    def place_config_inspector(self) -> Place:
        """Return a place to inspect this workspace task configuration."""
        from debusine.server.scopes import urlconf_scope

        with urlconf_scope(self.instance.scope.name):
            url = reverse(
                "workspaces:task_configuration_inspector",
                kwargs={"wname": self.instance.name},
            )

        return Place(
            title=f"Task configuration inspector"
            f" for {self.instance.name} workspace",
            url=url,
        )

    @cached_property
    def place_update_inheritance(self) -> Place:
        """Return a place to update this workspace's inheritance."""
        from debusine.server.scopes import urlconf_scope

        with urlconf_scope(self.instance.scope.name):
            url = reverse(
                "workspaces:update_inheritance",
                kwargs={"wname": self.instance.name},
            )

        return Place(
            title=f"Configure inheritance of workspace {self.instance.name}",
            url=url,
        )

    @cached_property
    def place_workflow_list(self) -> Place:
        """Return a place to list this workspace's workflows."""
        from debusine.server.scopes import urlconf_scope

        with urlconf_scope(self.instance.scope.name):
            url = reverse(
                "workspaces:workflows:list",
                kwargs={"wname": self.instance.name},
            )

        return Place(
            title=f"{self.instance.name} workflows",
            url=url,
        )

    @cached_property
    def place_collection_list(self) -> Place:
        """Return a place to list this workspace's collections."""
        from debusine.server.scopes import urlconf_scope

        with urlconf_scope(self.instance.scope.name):
            url = reverse(
                "workspaces:collections:list",
                kwargs={"wname": self.instance.name},
            )

        return Place(
            title=f"{self.instance.name} collections",
            url=url,
        )

    @cached_property
    def candidate_parents(self) -> list[Workspace]:
        """Return a list of workspaces that can be parents to this one."""
        base_qs = Workspace.objects.select_related("scope")
        allowed = base_qs.can_display(context.user).exclude(pk=self.instance.pk)
        allowed = allowed.union(
            base_qs.filter(pk__in=self.instance.chain_parents.values("parent"))
        )
        return list(allowed.order_by("scope", "name"))
