/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.java;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.Factory;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * Factory for elements of the Java namespace.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class JavaFactory {

	private JavaFactory() {
	}

	/**
	 * Create a factory for {@link JavaVersion} elements.
	 *
	 * @param version the Java version
	 * @return JavaVersion factory
	 */
	public static Factory<JavaVersion> javaVersion(String version) {
		return context -> new JavaVersion(context, version);
	}

	/**
	 * Create a factory for {@link FileEncoding} elements.
	 *
	 * @param fileEncoding the file encoding
	 * @return FileEncoding factory
	 */
	public static Factory<FileEncoding> fileEncoding(String fileEncoding) {
		return context -> new FileEncoding(context, fileEncoding);
	}

	/**
	 * Create a factory for {@link HeapSize} elements.
	 *
	 * @return HeapSize factory
	 */
	public static Factory<HeapSize> heapSize() {
		return HeapSize::new;
	}

	/**
	 * Create a factory for {@link ClassSource} elements.
	 *
	 * @param className the source class name
	 * @return ClassSource factory
	 */
	public static Factory<ClassSource> classSource(String className) {
		return context -> new ClassSource(context).withClassName(className);
	}

	/**
	 * Create a factory for {@link MethodSource} elements.
	 *
	 * @param className the source class name
	 * @param methodName the source method name
	 * @return MethodSource factory
	 */
	public static Factory<MethodSource> methodSource(String className, String methodName) {
		return context -> new MethodSource(context).withClassName(className).withMethodName(methodName);
	}

	/**
	 * Create a factory for {@link ClasspathResourceSource} elements.
	 *
	 * @param resourceName the source resource name
	 * @return ClasspathResourceSource factory
	 */
	public static Factory<ClasspathResourceSource> classpathResourceSource(String resourceName) {
		return context -> new ClasspathResourceSource(context).withResourceName(resourceName);
	}

	/**
	 * Create a factory for {@link PackageSource} elements.
	 *
	 * @param packageName the source package name
	 * @return PackageSource factory
	 */
	public static Factory<PackageSource> packageSource(String packageName) {
		return context -> new PackageSource(context).withName(packageName);
	}

	/**
	 * Create a factory for {@link Throwable} elements.
	 *
	 * @param throwable the throwable
	 * @return Throwable factory
	 */
	public static Factory<Throwable> throwable(java.lang.Throwable throwable) {
		return context -> new Throwable(context).withThrowable(throwable);
	}
}
