/*
* Copyright (C) 2023, KylinSoft Co., Ltd.
*               2010-2016 LXQt team.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301, USA.
**/
#include "modulemanager.h"
#include "ukuimodule.h"
#include "idlewatcher.h"
#include "ukuisessiondebug.h"
#include "xdgautostart.h"
#include "xdgdesktopfile.h"
#include "xdgdirs.h"

#include <QCoreApplication>
#include <QFileInfo>
#include <QStringList>
#include <QSettings>
#include <QStandardPaths>
#include <QDebug>
#include <QGSettings/QGSettings>
// #include <QSoundEffect>
#include <QDBusInterface>
#include <QDBusPendingCall>
#include <QDBusReply>
#include <QDir>
#include <QDBusReply>
/* qt会将glib里的signals成员识别为宏，所以取消该宏
 * 后面如果用到signals时，使用Q_SIGNALS代替即可
 **/
#ifdef signals
#undef signals
#endif

#define SESSION_REQUIRED_COMPONENTS "org.ukui.session.required-components"
#define SESSION_REQUIRED_COMPONENTS_PATH "/org/ukui/desktop/session/required-components/"

#define SSWND_DBUS_SERVICE     "org.ukui.ScreenSaverWnd"
#define SSWND_DBUS_PATH        "/"
#define SSWND_DBUS_INTERFACE   "org.ukui.ScreenSaverWnd"

#define FD_DBUS_SERVICE     "org.freedesktop.DBus"
#define FD_DBUS_PATH        "/org/freedesktop/DBus"
#define FD_DBUS_INTERFACE   "org.freedesktop.DBus"

std::map<const QString, int> ModuleManager::m_startupMap = {};

void ModuleManager::playBootMusic(bool arg)
{
    if( arg ) return;
    //set default value of whether boot-music is opened
    bool play_music = true;
    if (QGSettings::isSchemaInstalled("org.ukui.session")) {
        QGSettings *gset = new QGSettings("org.ukui.session", "/org/ukui/desktop/session/", this);
        if (gset == NULL) {
            qDebug() << "QGSettings init error";
            delete gset;
            return;
        }
        QString xdg_session_type = qgetenv("XDG_SESSION_TYPE");
        if (arg) {
            play_music = gset->get("startup-music").toBool();
            if (play_music) {
                if (xdg_session_type == "wayland") {
                    //非华为机器没有声音放大器，默认不设置音量大小
                    QProcess::startDetached("paplay /usr/share/ukui/ukui-session-manager/startup.wav");
                } else {
                    QProcess::startDetached("aplay  /usr/share/ukui/ukui-session-manager/startup.wav");
                }
            }
        } else {
            play_music = gset->get("weakup-music").toBool();
            if (play_music) {
                if (xdg_session_type == "wayland") {
                    QProcess::startDetached("paplay /usr/share/ukui/ukui-session-manager/weakup.wav");
                } else {
                    QProcess::startDetached("aplay /usr/share/ukui/ukui-session-manager/weakup.wav");
                }
            }
        }
    }
}

//睡眠转休眠判断
bool isSuspendToHibernate()
{
    QString path = "/sys/power/wakeup_reason";
    if (!QFile::exists(path)) {
        return false;
    }
    QFile wakeup_reason_file(path);
    wakeup_reason_file.open(QIODevice::ReadOnly | QIODevice::Text);
    QString wakeupReason = wakeup_reason_file.readAll().simplified();
    wakeup_reason_file.close();
    if (QString::compare(wakeupReason,"0xaa") == 0) {
        qDebug()<< "Suspend to Hibernate, wakeup_reason is "<<wakeupReason;
        return true;
    }
    return false;
}

ModuleManager::ModuleManager( QObject* parent) : QObject(parent)
{
    QDBusConnection::systemBus().connect(QString("org.freedesktop.login1"),
                                         QString("/org/freedesktop/login1"),
                                         QString("org.freedesktop.login1.Manager"),
                                         QString("PrepareForSleep"), this, SLOT(weakup(bool)));
    constructStartupList();

    ChkScreenLockStartup();
}

void ModuleManager::weakup(bool arg)
{
    if (arg) {
        qDebug() << "准备执行睡眠休眠";
    } else {
        qDebug() << "从睡眠休眠中唤醒";
        if (!isSuspendToHibernate()) {
            qDebug() << "播放唤醒音乐";
            playBootMusic(false);
        }
    }
}

ModuleManager::~ModuleManager()
{
    ModulesMapIterator i(mNameMap);
    while (i.hasNext()) {
        i.next();

        auto p = i.value();
        disconnect(p, SIGNAL(finished(int, QProcess::ExitStatus)), nullptr, nullptr);

        delete p;
        mNameMap[i.key()] = nullptr;
    }
}

void ModuleManager::constructStartupList()
{
    const QByteArray id(SESSION_REQUIRED_COMPONENTS);
    QString window_manager;
    QString panel;
    QString file_manager;
    QString wm_notfound;
    if (QGSettings::isSchemaInstalled(id)) {
        const QGSettings *gs = new QGSettings(SESSION_REQUIRED_COMPONENTS, SESSION_REQUIRED_COMPONENTS_PATH, this);
        if (gs == NULL) {
            qDebug() << "QGSettings init error";
            return;
        }
        window_manager = gs->get("windowmanager").toString() + ".desktop";
        panel = gs->get("panel").toString() + ".desktop";
        file_manager = gs->get("filemanager").toString() + ".desktop";
        wm_notfound = gs->get("windowmanager").toString();
    } else {
        //gsetting安装失败，或无法获取，设置默认值
        qDebug() << "从gsettings 中或取值失败，设置默认值";
        window_manager = "ukwm.desktop";
        panel = "ukui-panel.desktop";
        file_manager = "peony-qt-desktop.desktop";
    }

    QString xdg_session_type = qgetenv("XDG_SESSION_TYPE");
    if (xdg_session_type == "wayland") {
        isWayland = true;
    }

    QStringList desktop_paths;
    desktop_paths << "/usr/share/applications";
    desktop_paths << "/etc/xdg/autostart";
    bool panel_found = false;
    bool fm_found = false;
    bool wm_found = false;

    //const auto files = XdgAutoStart::desktopFileList(desktop_paths, false);
    for (const QString &dirName : const_cast<const QStringList&>(desktop_paths)) {
        QDir dir(dirName);
        if (!dir.exists()) continue;
        const QFileInfoList files = dir.entryInfoList(QStringList(QLatin1String("*.desktop")), QDir::Files | QDir::Readable);
        for (const QFileInfo &fi : files) {
            if (fi.fileName() == panel) {
                mPanel.load(fi.absoluteFilePath());
                panel_found = true;
                qDebug() << "panel has been found";
            }
            if (fi.fileName() == file_manager) {
                mFileManager.load(fi.absoluteFilePath());
                fm_found = true;
                qDebug() << "filemanager has been found";
            }
            if (fi.fileName() == window_manager) {
                mWindowManager.load(fi.absoluteFilePath());
                wm_found = true;
                qDebug() << "windowmanager has been found";
            }

            if (fm_found && panel_found && wm_found)
                break;
        }
    }

    if (!panel_found || !fm_found || !wm_found) isDirectInstall = true;
    if (isDirectInstall) {
        wmFound = wm_found;
    }

    if (wm_found == false) {
        QFileInfo check_ukwm("/usr/share/applications/ukwm.desktop");
        QFileInfo check_ukuikwin("/usr/share/applications/ukui-kwin.desktop");
        if (check_ukwm.exists()) {
            window_manager = "ukwm.desktop";
            mWindowManager.load("/usr/share/applications/ukwm.desktop");
        } else if (check_ukuikwin.exists()) {
            window_manager = "ukui-kwin.desktop";
            mWindowManager.load("/usr/share/applications/ukui-kwin.desktop");
        }
        wm_found = true;
    }

    //配置文件所给的窗口管理器找不到.desktop文件时，将所给QString设为可执行命令，创建一个desktop文件赋给mWindowManager
//    if (wm_found == false) {
//        mWindowManager = XdgDesktopFile(XdgDesktopFile::ApplicationType,"window-manager", wm_notfound);
//        qDebug() << "windowmanager has been created";
//    }

    QString desktop_phase = "X-UKUI-Autostart-Phase";
    QString desktop_type = "Type";
    //设置excludeHidden为true，判断所有desktop文件的Hidden值，若为true，则将其从自启列表中去掉
    const XdgDesktopFileList all_file_list = XdgAutoStart::desktopFileList(true);
    for (XdgDesktopFileList::const_iterator i = all_file_list.constBegin(); i != all_file_list.constEnd(); ++i) {
        QString filePath = i->fileName();
        QString filename = QFileInfo(filePath).fileName();
        if (filename == panel || filename == file_manager || filename == window_manager) {
            continue;
        }
        const XdgDesktopFile file = *i;
        if (i->contains(desktop_phase)) {
            QStringList s1 = file.value(desktop_phase).toString().split(QLatin1Char(';'));
            if (s1.contains("Initialization")) {
                mInitialization << file;
            } else if (s1.contains("Desktop")) {
                mDesktop << file;
            } else if (s1.contains("Application")) {
                //用户设置的自启应用要单独判断，有可能会存在问题，但暂时想不到更好的判断方法
                if (filePath.contains(".config/autostart")) {
                    m_userAutoStartApp << file;
                } else {
                    mApplication << file;
                }
            }
        } else if (i->contains(desktop_type)) {
            QStringList s2 = file.value(desktop_type).toString().split(QLatin1Char(';'));
            if (s2.contains("Application")) {
                if (filePath.contains(".config/autostart")) {
                    m_userAutoStartApp << file;
                } else {
                    mApplication << file;
                }
            }
        }
    }

    QStringList force_app_paths;
    force_app_paths << "/usr/share/ukui/applications";
    const XdgDesktopFileList force_file_list = XdgAutoStart::desktopFileList(force_app_paths, true);
    for (XdgDesktopFileList::const_iterator i = force_file_list.constBegin(); i != force_file_list.constEnd(); ++i) {
        qDebug() << (*i).fileName();
        mForceApplication << *i;
    }
}

/* Startup Phare:
 *  Initialization
 *  WindowManager
 *  Panel
 *  FileManager
 *  Desktop
 *  Application
 *
 */

bool ModuleManager::startModuleTimer(QTimer *timer, int i)
{
    timer->setSingleShot(true);
    connect(timer, SIGNAL(timeout()), this, SLOT(timeup()));
    timer->start(i * 1000);
    return true;
}

void ModuleManager::startupfinished(const QString &appName, const QString &string)
{
    qDebug() << "moudle :" + appName + " startup finished, and it want to say " + string;
    if (appName == "ukui-settings-daemon") {
        tusd->stop();
        emit usdfinished();
        return;
    }
    if (appName == "ukui-kwin") {
        twm->stop();
        isWMStarted = true;
        emit wmfinished();
        return;
    }
    if (appName == "ukui-panel") {
        tpanel->stop();
        isPanelStarted = true;
        emit panelfinished();
        return;
    }
    if (appName == "peony-qt-desktop") {
        tdesktop->stop();
        isDesktopStarted = true;
        emit desktopfinished();
        return;
    }
}

void ModuleManager::timeup()
{
    QTimer *time_out = qobject_cast<QTimer*>(sender());
    if (time_out == tusd) {
        qDebug() << "usd超时";
        emit usdfinished();
        return;
    }
    if (time_out == twm) {
        qDebug() <<"wm超时";
        isWMStarted = true;
        emit wmfinished();
        return;
    }
    if (time_out == tpanel) {
        qDebug() << "panel超时";
        isPanelStarted = true;
        emit panelfinished();
        return;
    }
    if (time_out == tdesktop) {
        qDebug() << "peony-qt-desktop超时";
        isDesktopStarted = true;
        emit desktopfinished();
        return;
    }
}

void ModuleManager::startCompsite()
{
    if (isWayland) return;

    qDebug() << "Enter:: startCompsite";
    if (!isPanelStarted || !isDesktopStarted || !isWMStarted) return;//  || !isWMStarted

    if (isCompsiteStarted) return;
    isCompsiteStarted = true;

    timerUpdate();

/*    //按kwin要求，kwin现在启动后自动开混成
    // start composite
    QDBusInterface dbus("org.ukui.KWin", "/Compositor", "org.ukui.kwin.Compositing", QDBusConnection::sessionBus());

    if (!dbus.isValid()) {
        qWarning() << "dbusCall: QDBusInterface is invalid ; kwin do not exit!";
        timerUpdate();
    } else {
        qDebug() << "Start composite";
        dbus.call("resume");

        timerUpdate();
    }*/
}

void ModuleManager::startup()
{
    //直接安装进入的流程
    const QFile file_installer("/etc/xdg/autostart/kylin-os-installer.desktop");
    if (file_installer.exists() && isDirectInstall) {
        if (wmFound) {
            isPanelStarted   = true;
            isDesktopStarted = true;
            connect(this, &ModuleManager::wmfinished, [=](){ startCompsite(); });

//            startProcess(mWindowManager, true);
            startModuleTimer(twm, 18);
        } else {
            timerUpdate();
        }

        return;
    }

    if (isWayland) {
        startProcess("hwaudioservice", true);
    }

    if (!isWayland) {
        connect(this, &ModuleManager::panelfinished, [=](){ startCompsite(); });
        connect(this, &ModuleManager::wmfinished, [=](){ startCompsite(); });
        connect(this, &ModuleManager::desktopfinished, [=](){ startCompsite(); });

//        dostartwm();
    } else {
        QTimer::singleShot(500, this, SLOT(timerUpdate()));
    }

    qDebug() << "Start Initialization app: ";
    for (XdgDesktopFileList::const_iterator i = mInitialization.constBegin(); i != mInitialization.constEnd(); ++i) {
        startProcess(*i, true);
    }
    startModuleTimer(twm,3);

    startProcess(mPanel, true);
    startModuleTimer(tpanel,5);

    startProcess(mFileManager, true);
    startModuleTimer(tdesktop,5);

    if (!isWayland) {
        qDebug() << "Start desktop: ";
        for (XdgDesktopFileList::const_iterator i = mDesktop.constBegin(); i != mDesktop.constEnd(); ++i) {
            startProcess(*i, true);
        }
    }
}

void ModuleManager::dostartwm()
{
    qDebug() << "Start window manager: " << mWindowManager.name();
    if (mWindowManager.name() == "UKUI-KWin") {
        startProcess(mWindowManager, true);
        startModuleTimer(twm, 18);
    } else {
        startProcess(mWindowManager, true);
        isWMStarted = true;
    }
}

void ModuleManager::timerUpdate()
{
    QTimer::singleShot(500, this, [&](){
        emit finished();
    });

    if (isWayland) {
        qDebug() << "Start desktop: ";
        for (XdgDesktopFileList::const_iterator i = mDesktop.constBegin(); i != mDesktop.constEnd(); ++i) {
            startProcess(*i, true);
        }
    }

    qDebug() << "Start application: ";
    QFile file_nm("/etc/xdg/autostart/kylin-nm.desktop");
    QFile file_sogou("/usr/bin/sogouImeService");
    for (XdgDesktopFileList::const_iterator i = mApplication.constBegin(); i != mApplication.constEnd(); ++i) {
        qDebug() << i->fileName();
        if (i->fileName() == "/etc/xdg/autostart/nm-applet.desktop" && file_nm.exists()) {
            qDebug() << "the kylin-nm exist so the nm-applet will not start";
            continue;
        }
        if (i->fileName() == "/etc/xdg/autostart/fcitx-qimpanel-autostart.desktop" && file_sogou.exists()) {
            qDebug() << "the sogouImeService exist so the fcitx-ui-qimpanel will not start";
            continue;
        }
        startProcess(*i, false);
    }

    qDebug() << "Start force application: ";
    for (XdgDesktopFileList::const_iterator i = mForceApplication.constBegin(); i != mForceApplication.constEnd(); ++i) {
        startProcess(*i, true);
    }

    //启动用户自定义的开机自启应用
    for (XdgDesktopFileList::const_iterator i = m_userAutoStartApp.constBegin(); i != m_userAutoStartApp.constEnd(); ++i) {
        qDebug() << "start user's autostart application " << i->fileName();
        openAppWithAppManager(*i);
    }

    if (QGSettings::isSchemaInstalled("org.ukui.session")) {
        QGSettings *gset = new QGSettings("org.ukui.session", "/org/ukui/desktop/session/", this);
        bool restoreSession = gset->get("restore-session").toBool();
        if (restoreSession) {
            //此处设置800ms延迟的原因是，当程序运行到这里时，已经启动了很多应用，根据测试，所有这些应用启动完并且完成与xmspserver的通信一般就需要800ms，如果这里不等待800ms，
            //然后直接调用xsmpserver的restoreSession接口，则xsmpserver的主循环会被用来启动那些待恢复的应用,从而无暇顾及其他应用的信号请求，造成阻塞，影响整个启动流程
            //之前觉得把xsmpserver分离出去可以解决这个问题，现在看来还是不行，因为ukuismserver还是要跟ukui-session交互并且判断是否该应用是否已经已经启动过，真正要解决这个
            //问题，还是要把启动阶段划分好，在restoreSession阶段保证就不会有别的应用干扰。
            QTimer::singleShot(800, [](){
                //调用ukuismserver恢复会话的接口
                qDebug(UKUI_SESSION) << "began restore session";
                QDBusInterface ukuismserverFace("org.ukui.ukuismserver",
                                                "/UKUISMServer",
                                                "org.ukui.ukuismserver",
                                                QDBusConnection::sessionBus());

                if (!ukuismserverFace.isValid()) {
                    qDebug() << "ukuisminterface not valid";
                    return;
                }

                ukuismserverFace.call(QDBus::CallMode::NoBlock, "restoreSession");
            });
        }
    }
}

void ModuleManager::startProcess(const XdgDesktopFile &file, bool required)
{
    QStringList args = file.expandExecString();
    if (args.isEmpty()) {
        qWarning() << "Wrong desktop file: " << file.fileName();
        return;
    }

    QString name = QFileInfo(file.fileName()).fileName();
    if (!mNameMap.contains(name)) {
        UkuiModule *proc = new UkuiModule(file, this);
        connect(proc, &UkuiModule::moduleStateChanged, this, &ModuleManager::moduleStateChanged);
        proc->startUKUIModule();

        mNameMap[name] = proc;

        if (required || autoRestart(file)) {
            connect(proc, SIGNAL(finished(int, QProcess::ExitStatus)),
                    this, SLOT(restartModules(int, QProcess::ExitStatus)));
        }
    }
}

void ModuleManager::startProcess(const QString& name, bool required)
{
    QString desktop_name = name + ".desktop";
    QStringList desktop_paths;
    desktop_paths << "/usr/share/applications";

    const auto files = XdgAutoStart::desktopFileList(desktop_paths, false);
    for (const XdgDesktopFile& file : files) {
        if (QFileInfo(file.fileName()).fileName() == desktop_name) {
            startProcess(file, required);
            return;
        }
    }
}

void ModuleManager::stopProcess(const QString& name)
{
     if (mNameMap.contains(name)) {
         mNameMap[name]->terminate();
     }
}

bool ModuleManager::startProcess(const QString &name, const QStringList &args)
{
    if (isProgramStarted(std::move(name))) {
        return false;
    }

    //这个接口供外部应用通过session打开应用使用，需要使用那个进程非正常死亡再拉起机制吗？
    QProcess *proc = new QProcess(this);
    proc->start(name, args);
    proc->closeReadChannel(QProcess::StandardOutput);
    proc->closeReadChannel(QProcess::StandardError);
    m_startupMap[std::forward<const QString>(name)] = 1;

    return true;
}

bool ModuleManager::nativeEventFilter(const QByteArray &eventType, void *message, long *result)
{
    if (eventType != "xcb_generic_event_t") // We only want to handle XCB events
        return false;

    return false;
}

void ModuleManager::insertStartupList(QString &&str)
{
    if (!str.isEmpty()) {
        m_startupMap[std::forward<QString>(str)] = 1;
    }
}

bool ModuleManager::isProgramStarted(const QString &&str)
{
    //需要对str做处理，取出可执行程序名，str一般是一个完整的二进制路径，所以既需要判断二进制路径也要判断二进制名
    QString appName = str.mid(str.lastIndexOf(QDir::separator()) + 1);
    auto itFullPath = m_startupMap.find(str);
    auto itName = m_startupMap.find(appName);
    qDebug() << "judge if start before " << appName << " " << str;

    if (itFullPath != m_startupMap.end() || itName != m_startupMap.end()) {
        qDebug() << str << "has start up before";
        return true;
    }

    return false;
}

bool ModuleManager::autoRestart(const XdgDesktopFile &file)
{
    QString auto_restart = "X-UKUI-AutoRestart";
    return file.value(auto_restart).toBool();
}

bool ModuleManager::openAppWithAppManager(const XdgDesktopFile &file)
{
    QDBusInterface face("com.kylin.AppManager",
                        "/com/kylin/AppManager",
                        "com.kylin.AppManager",
                        QDBusConnection::sessionBus());
    if (!face.isValid()) {
        qCDebug(UKUI_SESSION()) << "AppManager interface not valid";
        return false;
    }

    QString name = file.fileName();
    QVariant desktopFilePath(name);
    QDBusReply<bool> reply = face.call("LaunchApp", desktopFilePath);
    if (reply.isValid()) {
        return reply.value();
    }

    return true;
}

void ModuleManager::restartModules(int /*exitCode*/, QProcess::ExitStatus exitStatus)
{
    if (isShutingDown) {
        qDebug() << "Logout phase, don't Restart";
        return;
    }

    UkuiModule *proc = qobject_cast<UkuiModule*>(sender());

    if (nullptr == proc) {
        qWarning() << "Got an invalid (null) module to restart, Ignoring it";
        return;
    }

    //需要做出修改
    if (proc->restartNum > 10) {
        mNameMap.remove(proc->fileName);
        disconnect(proc, SIGNAL(finished(int, QProcess::ExitStatus)), nullptr, nullptr);
        proc->deleteLater();
        return;
    }

    if (!proc->isTerminating()) {
        //根据退出码来判断程序是否属于异常退出。
        QString procName = proc->file.name();
        if (proc->exitCode() == 0) {
            qDebug() << "Process" << procName << "(" << proc << ") exited correctly. " << "With the exitcode = " << proc->exitCode() << ",exitStatus = " << exitStatus;
        } else {
            qDebug() << "Process" << procName << "(" << proc << ") has to be restarted. " << "With the exitcode = " << proc->exitCode() << ",exitStatus = " << exitStatus;
            proc->startUKUIModule();
            proc->restartNum++;
            return;
        }
    }
    mNameMap.remove(proc->fileName);
    proc->deleteLater();
}

bool ModuleManager::getIsShutingDown() const
{
    return isShutingDown;
}

void ModuleManager::setIsShutingDown(bool value)
{
    isShutingDown = value;
}

void ModuleManager::logout(bool doExit)
{
    // /org/freedesktop/login1/session/self 和 /org/freedesktop/login1/session/auto
    //有什么区别
    QDBusInterface face("org.freedesktop.login1",\
                        "/org/freedesktop/login1/session/self",\
                        "org.freedesktop.login1.Session",\
                        QDBusConnection::systemBus());

    if (doExit) {
        face.call("Terminate");
        exit(0);
    }
}

void ModuleManager::ChkScreenLockStartup()
{
    QDBusInterface *iface = new QDBusInterface(SSWND_DBUS_SERVICE,
                                               SSWND_DBUS_PATH,
                                               SSWND_DBUS_INTERFACE,
                                               QDBusConnection::sessionBus(),
                                               this);
    if (iface && iface->isValid()) {
        QDBusReply<bool> reply =  iface->call("IsStartupMode");
        if (reply.isValid()) {
            qDebug()<<"ScreenLockStartup:"<<reply.value();
            isScreenLockStartup = reply.value();
        }
        //connect(iface, SIGNAL(StartupModeChanged(bool)), this, SLOT(onSLStartupModeChanged(bool)));
    }
    // 监听锁屏退出
    if (isScreenLockStartup) {
        QDBusInterface *dbusService = new QDBusInterface(FD_DBUS_SERVICE,
                                         FD_DBUS_PATH,
                                         FD_DBUS_INTERFACE,
                                         QDBusConnection::sessionBus(),
                                         this);
        connect(dbusService, SIGNAL(NameOwnerChanged(QString, QString, QString)),
                this, SLOT(onServiceStatusChanged(QString, QString, QString)));
    }
}

void ModuleManager::onServiceStatusChanged(const QString &name,
                                           const QString &oldOwner,
                                           const QString &newOwner)
{
    if(name == SSWND_DBUS_SERVICE)
    {
        bool isScreenLockStarted = !newOwner.isEmpty();
        qDebug() << "ScreenLock service status changed:"
                 << (isScreenLockStarted ? "activate" : "inactivate");
        if (!isScreenLockStarted) {
            onSLStartupModeChanged(isScreenLockStarted);
        }
    }
}

void ModuleManager::onSLStartupModeChanged(bool isStartup)
{
    static bool isPlayBootMusicDone = false;
    isScreenLockStartup = isStartup;
    qDebug()<<"onSLStartupModeChanged:"<<isScreenLockStartup;
    if (!isScreenLockStartup && !isPlayBootMusicDone) {
        isPlayBootMusicDone = true;
        playBootMusic(true);
    }
}


#include "modulemanager.moc"
